package chatclient;

import java.awt.EventQueue;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.border.EmptyBorder;
import java.awt.Toolkit;

import javax.swing.GroupLayout;
import javax.swing.GroupLayout.Alignment;
import javax.swing.BoxLayout;
import java.awt.BorderLayout;
import javax.swing.JToolBar;
import javax.swing.JScrollPane;
import javax.swing.JButton;
import javax.swing.JTextField;
import javax.swing.border.EtchedBorder;

import chatclient.ServerConnector.ApiResponse;

import javax.swing.ScrollPaneConstants;
import javax.swing.SwingUtilities;
import javax.swing.Timer;
import javax.swing.JToggleButton;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import javax.swing.JProgressBar;
import javax.swing.JScrollBar;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import javax.swing.UIManager;

public class MainFrame extends JFrame {

	private static final long serialVersionUID = 1L;
	private static MainFrame INSTANCE;

	private ServerConnector connector = null;
	private String userId = null;

	private JPanel contentPane;
	private JTextField tfUserMsg;
	private JPanel pnlChat;
	private JButton btnSend;
	private JToggleButton tglConnect;
	private JScrollPane spChat;
	private JProgressBar pbWait;

	/**
	 *
	 * @param id
	 */
	public static void onRemoteMsgLinkClicked(String id) {
		if(id != null && !id.isBlank()) {
			if(id.startsWith("btn_snd_")) {
				String response = id.substring(8).replaceAll("_", " ");
				INSTANCE.tfUserMsg.setText(response);
				INSTANCE.btnSend.doClick();
			}
		}
	}

	/**
	 * Scrolls the chat window to the bottom
	 */
	private void scrollToBottom() {
		Timer timer = new Timer(10, null);
		ActionListener al = new ActionListener() {
			float steps = -1.0f;
			@Override
			public void actionPerformed(ActionEvent e) {
				JScrollBar bar = spChat.getVerticalScrollBar();
				int max = bar.getMaximum() - bar.getVisibleAmount();
				if(steps < 0.0f)
					steps = (max - bar.getValue()) / 10;
				bar.setValue(bar.getValue() + Math.round(steps*=0.915f));
				if(steps <= 1.0f)
					steps = 1.0f;
				if(bar.getValue() >= max)
					timer.stop();
			}
		};
		timer.addActionListener(al);
		timer.start();
	}

	/**
	 * Create the frame.
	 */
	public MainFrame() {
		INSTANCE = this;
		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		setTitle("Chat4Us - Client example");
		setBounds(100, 100, 420, 600);
		contentPane = new JPanel();
		contentPane.setBorder(new EmptyBorder(5, 5, 5, 5));
		setContentPane(contentPane);
		contentPane.setLayout(new BorderLayout(0, 0));

		JToolBar toolBar = new JToolBar();
		toolBar.setBorder(new EtchedBorder(EtchedBorder.LOWERED, null, null));
		contentPane.add(toolBar, BorderLayout.NORTH);

		tglConnect = new JToggleButton("Connect");
		tglConnect.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if(tglConnect.isSelected()) {
					ConnectDialog dlg = new ConnectDialog();
					dlg.setLocationRelativeTo(MainFrame.this);
					dlg.setVisible(true);
					if(!dlg.isCancelled()) {
						pnlChat.removeAll();
						connector = new ServerConnector(dlg.getKey1(), dlg.getKey2(), "https://" + dlg.getHost() + ":" + dlg.getPort());
						if(connector.login()) {
							userId = Helper.getRandomString(10);
							ApiResponse response = connector.letsChat(userId);
							if(response != null && response.isSuccess()) {
								String[] messages = response.getMessages();
								for(String message : messages) {
									IMessagePanel panel = new RemoteMessage();
									panel.setMessage(message, System.currentTimeMillis());
									pnlChat.add((JPanel)panel);
									pnlChat.revalidate();
									pnlChat.repaint();
								}
								if(!response.isChatEnded()) {
									btnSend.setEnabled(true);
									tfUserMsg.setEnabled(true);
									tfUserMsg.requestFocusInWindow();
								}
							}
						} else {
							JOptionPane.showMessageDialog(MainFrame.this, "Connection failed!\nPlease try again later...", "Connection failed!", JOptionPane.ERROR_MESSAGE);
							tglConnect.setSelected(false);
						}
					}
					dlg.dispose();
				} else {
					pnlChat.setEnabled(true);
					tfUserMsg.setEnabled(false);
					btnSend.setEnabled(false);
					pbWait.setVisible(false);
					JOptionPane.showMessageDialog(MainFrame.this, "You've been disconnected!\nPlease try again later...", "Disconnected!", JOptionPane.ERROR_MESSAGE);
				}
			}
		});
		tglConnect.setFocusable(false);
		toolBar.add(tglConnect);
		toolBar.addSeparator();

		pbWait = new JProgressBar();
		pbWait.setVisible(false);
		pbWait.setIndeterminate(true);
		toolBar.add(pbWait);

		JPanel panel = new JPanel();
		panel.setBorder(new EtchedBorder(EtchedBorder.LOWERED, null, null));
		contentPane.add(panel, BorderLayout.SOUTH);

		tfUserMsg = new JTextField();
		tfUserMsg.addKeyListener(new KeyAdapter() {
			@Override
			public void keyReleased(KeyEvent e) {
				if(e.getKeyCode() == KeyEvent.VK_ENTER) {
					btnSend.doClick();
				}
			}
		});
		tfUserMsg.setEnabled(false);
		tfUserMsg.setColumns(10);

		btnSend = new JButton("Send");
		btnSend.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if(connector != null && connector.isLoggedIn()) {
					String msg = tfUserMsg.getText();
					if(!msg.isBlank()) {
						tfUserMsg.setText("");
						tfUserMsg.setEnabled(false);
						btnSend.setEnabled(false);
						IMessagePanel panel = new UserMessage();
						panel.setMessage(Helper.escapeHtml(msg), System.currentTimeMillis());
						pnlChat.add((JPanel)panel);
						pbWait.setVisible(true);
						pnlChat.revalidate();
						pnlChat.repaint();
						pnlChat.setEnabled(false);
						tfUserMsg.setEnabled(false);
						btnSend.setEnabled(false);
						pbWait.setVisible(true);
						Thread th = new Thread(() -> {
							ApiResponse response = connector.sendMessage(userId, msg);
							if(response != null) {
								handleServerAnswer(response);
								if(response.isChatbotWaiting()) {
									SwingUtilities.invokeLater(() -> {
										pnlChat.setEnabled(false);
										tfUserMsg.setEnabled(false);
										btnSend.setEnabled(false);
										pbWait.setVisible(true);
									});
									ApiResponse aiResponse = connector.sendMessage(userId, "...");
									handleServerAnswer(aiResponse);
								}
							} else { // Fatal error or disconnected
								SwingUtilities.invokeLater(() -> {
									tglConnect.setSelected(false); // Disconnect
								});
							}
						});
						th.start();
					} else Toolkit.getDefaultToolkit().beep();
				} else {
					tfUserMsg.setText("");
					tfUserMsg.setEnabled(false);
					btnSend.setEnabled(false);
				}
			}

			/**
			 *
			 * @param response
			 */
			private void handleServerAnswer(ApiResponse response) {
				if(response != null && response.isSuccess()) {
					String[] messages = response.getMessages();
					for(String message : messages) {
						if(!message.isBlank()) {
							SwingUtilities.invokeLater(() -> {
								IMessagePanel pnl;
								pnl = new RemoteMessage();
								pnl.setMessage(message, System.currentTimeMillis());
								pnl.setIcon("/" + response.getChatStateIconName() + ".png");
								pnlChat.add((JPanel)pnl);
							});
						}
					}
					if(!response.isChatEnded()) {
						SwingUtilities.invokeLater(() -> {
							pnlChat.setEnabled(true);
							btnSend.setEnabled(true);
							pbWait.setVisible(false);
							tfUserMsg.setEnabled(true);
							tfUserMsg.requestFocusInWindow();
							scrollToBottom();
						});
					} else {
						pnlChat.setEnabled(false);
						btnSend.setEnabled(false);
						pbWait.setVisible(false);
						tfUserMsg.setEnabled(false);
						tfUserMsg.requestFocusInWindow();
						scrollToBottom();
					}
					if(response.isChatbotWaiting()) {
						SwingUtilities.invokeLater(() -> {
							// Should send "..." back to server in order to connect to an agent/messenger.
							tfUserMsg.setText("...");
							btnSend.doClick();
						});
					}
					pnlChat.revalidate();
					pnlChat.repaint();
				} else if(response != null) {
					SwingUtilities.invokeLater(() -> {
						IMessagePanel pnl;
						pnl = new ErrorMessage();
						pnl.setMessage("Error while sending the message", System.currentTimeMillis());
						pnlChat.add((JPanel)pnl);
						pnlChat.revalidate();
						pnlChat.repaint();
					});
				}
			}
		});
		btnSend.setEnabled(false);
		GroupLayout gl_panel = new GroupLayout(panel);
		gl_panel.setHorizontalGroup(
			gl_panel.createParallelGroup(Alignment.LEADING)
				.addGroup(Alignment.TRAILING, gl_panel.createSequentialGroup()
					.addGap(1)
					.addComponent(tfUserMsg, GroupLayout.DEFAULT_SIZE, 318, Short.MAX_VALUE)
					.addGap(3)
					.addComponent(btnSend)
					.addGap(1))
		);
		gl_panel.setVerticalGroup(
			gl_panel.createParallelGroup(Alignment.LEADING)
				.addGroup(gl_panel.createSequentialGroup()
					.addGroup(gl_panel.createParallelGroup(Alignment.BASELINE)
						.addComponent(tfUserMsg, GroupLayout.DEFAULT_SIZE, 23, Short.MAX_VALUE)
						.addComponent(btnSend))
					.addGap(1))
		);
		panel.setLayout(gl_panel);

		spChat = new JScrollPane();
		spChat.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
		spChat.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
		contentPane.add(spChat, BorderLayout.CENTER);

		pnlChat = new JPanel();
		pnlChat.setBackground(UIManager.getColor("window"));
		spChat.setViewportView(pnlChat);
		pnlChat.setLayout(new BoxLayout(pnlChat, BoxLayout.Y_AXIS));

	}

	/**
	 * Launch the application.
	 */
	public static void main(String[] args) {
		EventQueue.invokeLater(new Runnable() {
			public void run() {
				try {
					MainFrame frame = new MainFrame();
					frame.setVisible(true);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		});
	}
}
