package io.github.emmrida.jvsplit;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.ArrayList;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.filechooser.FileNameExtensionFilter;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.formdev.flatlaf.FlatDarkLaf;

import uk.co.caprica.vlcj.factory.discovery.NativeDiscovery;
import uk.co.caprica.vlcj.player.base.MediaPlayer;
import uk.co.caprica.vlcj.player.base.MediaPlayerEventAdapter;
import uk.co.caprica.vlcj.player.component.EmbeddedMediaPlayerComponent;

/**
 * The primary application window for the Java Video Splitter.
 * <p>
 * This class acts as the central controller, managing:
 * <ul>
 * <li>The Video Player (VLCJ)</li>
 * <li>The Visual Timeline (TimelineComponent)</li>
 * <li>Project State (Load/Save JSON)</li>
 * <li>Export Logic (FFmpeg Wrapper)</li>
 * </ul>
 */
public class MainWindow extends JFrame {

    // --- State & Data ---
    /** The current data model loaded in the application. */
    private ProjectModel currentProject;
    
    /** The file path on disk where the current project is saved (null if new). */
    private File currentProjectFile = null;
    
    /** Flag indicating if the project has unsaved changes. */
    private boolean isDirty = false;
    
    /** JSON serializer/deserializer. */
    private final ObjectMapper jsonMapper = new ObjectMapper();

    // --- Preview Mode State ---
    private boolean isPreviewingAll = false;
    private int currentPreviewIndex = -1;

    // --- UI Components ---
    private final EmbeddedMediaPlayerComponent mediaPlayerComponent;
    private TimelineComponent timelineComponent;
    private JButton btnPlayPause;
    
    // --- Constants ---
    private static final String TITLE = "Java Video Splitter";

    /**
     * Constructs the Main Window and initializes all UI components and event listeners.
     */
    public MainWindow() {
        super(TITLE);
        setJMenuBar(createMenuBar());
        
        // 1. Setup Layout
        setLayout(new BorderLayout());
        mediaPlayerComponent = new EmbeddedMediaPlayerComponent();
        add(mediaPlayerComponent, BorderLayout.CENTER);
        
        // 2. Setup Control Panels
        JPanel southPanel = new JPanel();
        southPanel.setLayout(new BoxLayout(southPanel, BoxLayout.Y_AXIS));
        
        southPanel.add(createPlaybackToolbar());
        southPanel.add(createTimelinePanel());
        southPanel.add(createActionToolbar());
        
        add(southPanel, BorderLayout.SOUTH);
        
        // 3. Window Configuration
        setSize(800, 600);
        setLocationRelativeTo(null);
        setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
        
        // 4. Initialize Logic
        setupWindowEvents();
        setupMediaPlayerEvents();
        resetProjectState();
    }

    // ==========================================
    // UI Construction
    // ==========================================

    /**
     * Creates the main application menu bar.
     * @return Constructed JMenuBar
     */
    private javax.swing.JMenuBar createMenuBar() {
        javax.swing.JMenuBar menuBar = new javax.swing.JMenuBar();
        javax.swing.JMenu fileMenu = new javax.swing.JMenu("File");
        
        javax.swing.JMenuItem newItem = new javax.swing.JMenuItem("New Project");
        newItem.addActionListener(e -> actionNewProject());
        
        javax.swing.JMenuItem openItem = new javax.swing.JMenuItem("Open Project");
        openItem.addActionListener(e -> actionOpenProject());
        
        javax.swing.JMenuItem saveItem = new javax.swing.JMenuItem("Save Project");
        saveItem.addActionListener(e -> actionSaveProject(false)); // Silent save
        
        javax.swing.JMenuItem saveAsItem = new javax.swing.JMenuItem("Save Project As...");
        saveAsItem.addActionListener(e -> actionSaveProject(true)); // Always prompt

        javax.swing.JMenuItem exitItem = new javax.swing.JMenuItem("Exit");
        exitItem.addActionListener(e -> {
            if (checkSaveNeeded()) {
                mediaPlayerComponent.release();
                System.exit(0);
            }
        });

        fileMenu.add(newItem);
        fileMenu.add(openItem);
        fileMenu.add(saveItem);
        fileMenu.add(saveAsItem);
        fileMenu.addSeparator();
        fileMenu.add(exitItem);

        javax.swing.JMenu helpMenu = new javax.swing.JMenu("Help");
        javax.swing.JMenuItem guideItem = new javax.swing.JMenuItem("User Guide");
        guideItem.addActionListener(e -> showUserGuide());
        javax.swing.JMenuItem aboutItem = new javax.swing.JMenuItem("About");
        aboutItem.addActionListener(e -> showAboutDialog());

        helpMenu.add(guideItem);
        helpMenu.add(aboutItem);
        menuBar.add(fileMenu);
        menuBar.add(helpMenu);
        return menuBar;
    }

    /**
     * Creates the toolbar containing Play, Pause, and Seek buttons.
     */
    private JPanel createPlaybackToolbar() {
        JPanel panel = new JPanel(new FlowLayout(FlowLayout.CENTER, 5, 5));
        panel.setBorder(BorderFactory.createMatteBorder(1, 0, 0, 0, Color.DARK_GRAY));

        JButton btnStart = new JButton("⏮ Start");
        JButton btnBack5s = new JButton("⏪ -5s");
        JButton btnBack1s = new JButton("◀ -1s");
        JButton btnBack01s = new JButton("◁ -0.1s");
        btnPlayPause = new JButton("▶ Play"); 
        JButton btnFwd01s = new JButton("+0.1s ▷");
        JButton btnFwd1s = new JButton("+1s ▶");
        JButton btnFwd5s = new JButton("+5s ⏩");
        JButton btnEnd = new JButton("End ⏭");

        btnStart.addActionListener(e -> { 
            mediaPlayerComponent.mediaPlayer().controls().setTime(0); 
            stopPreviewMode(); 
        });
        
        btnBack5s.addActionListener(e -> skip(-5000));
        btnBack1s.addActionListener(e -> skip(-1000));
        btnBack01s.addActionListener(e -> skip(-100));
        btnPlayPause.addActionListener(e -> togglePlayPause());
        btnFwd01s.addActionListener(e -> skip(100));
        btnFwd1s.addActionListener(e -> skip(1000));
        btnFwd5s.addActionListener(e -> skip(5000));
        
        btnEnd.addActionListener(e -> {
            mediaPlayerComponent.mediaPlayer().controls().setTime(mediaPlayerComponent.mediaPlayer().status().length());
            stopPreviewMode();
        });

        panel.add(btnStart);
        panel.add(btnBack5s);
        panel.add(btnBack1s);
        panel.add(btnBack01s);
        panel.add(btnPlayPause);
        panel.add(btnFwd01s);
        panel.add(btnFwd1s);
        panel.add(btnFwd5s);
        panel.add(btnEnd);
        return panel;
    }

    /**
     * Initializes the custom Timeline Component and binds its callbacks.
     */
    private JPanel createTimelinePanel() {
        timelineComponent = new TimelineComponent(
            // Callback 1: User drags timeline -> Seek video
            seekTime -> {
                mediaPlayerComponent.mediaPlayer().controls().setTime(seekTime);
                stopPreviewMode(); 
            },
            // Callback 2: User requests to play a specific segment
            segment -> playSelection(segment),
            // Callback 3: User requests to play all segments sequentially
            () -> startPreviewAll(),
            // Callback 4: User requests to mark a cut at current position
            () -> markSelection()
        );
        
        JPanel wrapper = new JPanel(new BorderLayout());
        wrapper.setBorder(BorderFactory.createEmptyBorder(5, 10, 5, 10));
        wrapper.add(timelineComponent, BorderLayout.CENTER);
        return wrapper;
    }

    /**
     * Creates the bottom toolbar with Project and Export actions.
     */
    private JPanel createActionToolbar() {
        JPanel panel = new JPanel(new FlowLayout(FlowLayout.CENTER, 5, 5));
        panel.setBorder(BorderFactory.createMatteBorder(1, 0, 0, 0, Color.DARK_GRAY));

        JButton btnNew = new JButton("New Project");
        JButton btnOpen = new JButton("Open Project");
        JButton btnSave = new JButton("Save Project");
        JButton btnExportSingle = new JButton("Export Single File");
        JButton btnExportMulti = new JButton("Export Multi-Files");
        JButton btnMarkCut = new JButton("✂ Mark Selection");
        btnMarkCut.setForeground(new Color(255, 100, 100));

        btnNew.addActionListener(e -> actionNewProject());
        btnOpen.addActionListener(e -> actionOpenProject());
        btnSave.addActionListener(e -> actionSaveProject(false));
        btnExportSingle.addActionListener(e -> actionExport(true));
        btnExportMulti.addActionListener(e -> actionExport(false));
        btnMarkCut.addActionListener(e -> markSelection());

        panel.add(btnNew);
        panel.add(btnOpen);
        panel.add(btnSave);
        panel.add(Box.createHorizontalStrut(5));
        panel.add(btnMarkCut);
        panel.add(Box.createHorizontalStrut(5));
        panel.add(btnExportSingle);
        panel.add(btnExportMulti);
        return panel;
    }

    // ==========================================
    // Core Logic
    // ==========================================

    /**
     * Skips the video forward or backward by a specific amount.
     * Updates the timeline immediately (Optimistic Update).
     * @param deltaMs The amount of milliseconds to skip (negative for backward).
     */
    private void skip(long deltaMs) {
        MediaPlayer player = mediaPlayerComponent.mediaPlayer();
        long current = player.status().time();
        long length = player.status().length();
        long target = current + deltaMs;
        
        if (target < 0) target = 0;
        if (target > length) target = length;
        
        player.controls().setTime(target);
        timelineComponent.setCurrentTime(target); // Optimistic UI update
        stopPreviewMode();
    }
    
    /**
     * Toggles between Play and Pause states. 
     * Handles the "Replay" logic if video is at the end.
     */
    private void togglePlayPause() {
        MediaPlayer player = mediaPlayerComponent.mediaPlayer();
        
        // Check if stopped or finished
        if (player.status().state() == uk.co.caprica.vlcj.player.base.State.STOPPED || 
            player.status().time() >= player.status().length()) {
            player.controls().setTime(0);
            player.controls().play();
            return;
        }
        
        if (player.status().isPlaying()) player.controls().pause();
        else player.controls().play();
    }

    /**
     * Logic for marking a selection at the current cursor position.
     * Checks for collisions with existing segments.
     */
    private void markSelection() {
        long total = mediaPlayerComponent.mediaPlayer().status().length();
        long now = mediaPlayerComponent.mediaPlayer().status().time();
        
        // Default length: 5% of video (min 2s, max 30s)
        long duration = (long) (total * 0.05); 
        if (duration < 2000) duration = 2000;
        if (duration > 30000) duration = 30000;
        
        long start = now;
        long end = now + duration;
        if (end > total) end = total;

        // Collision Check
        if (timelineComponent.isOverlapping(start, end)) {
            JOptionPane.showMessageDialog(this, "Cannot mark here: Overlaps with existing segment.", "Overlap", JOptionPane.WARNING_MESSAGE);
            return;
        }
        
        timelineComponent.addSegment(start, end);
        isDirty = true;
        updateTitle();
    }

    private void playSelection(TimelineComponent.Segment segment) {
        stopPreviewMode(); 
        mediaPlayerComponent.mediaPlayer().controls().setTime(segment.start);
        mediaPlayerComponent.mediaPlayer().controls().play();
    }

    /**
     * Initiates the Sequential Preview Mode.
     * Plays all segments one after another.
     */
    private void startPreviewAll() {
        List<TimelineComponent.Segment> segs = timelineComponent.getSegments();
        if (segs.isEmpty()) {
            JOptionPane.showMessageDialog(this, "No segments to play!");
            return;
        }
        isPreviewingAll = true;
        currentPreviewIndex = 0;
        mediaPlayerComponent.mediaPlayer().controls().setTime(segs.get(0).start);
        mediaPlayerComponent.mediaPlayer().controls().play();
    }

    private void stopPreviewMode() {
        if (isPreviewingAll) {
            isPreviewingAll = false;
            currentPreviewIndex = -1;
        }
    }

    // ==========================================
    // Event Listeners
    // ==========================================

    private void setupWindowEvents() {
        addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                if (checkSaveNeeded()) {
                    mediaPlayerComponent.release();
                    System.exit(0);
                }
            }
        });
    }

    /**
     * Configures listeners for VLCJ events.
     * Ensures all UI updates happen on the Swing Event Dispatch Thread (EDT).
     */
    private void setupMediaPlayerEvents() {
        mediaPlayerComponent.mediaPlayer().events().addMediaPlayerEventListener(new MediaPlayerEventAdapter() {
            @Override
            public void lengthChanged(MediaPlayer mediaPlayer, long newLength) {
                // Wait for video load to complete, then populate segments
                SwingUtilities.invokeLater(() -> {
                    timelineComponent.setTotalDuration(newLength);
                    if (currentProject != null && currentProject.segments != null) {
                        timelineComponent.clearSegments(); 
                        for (ProjectModel.Segment seg : currentProject.segments) {
                            timelineComponent.addSegment(seg.start, seg.end);
                        }
                        timelineComponent.repaint();
                    }
                });
            }

            @Override
            public void timeChanged(MediaPlayer mediaPlayer, long newTime) {
                SwingUtilities.invokeLater(() -> timelineComponent.setCurrentTime(newTime));
                
                // Handle Sequential Playback Logic
                if (isPreviewingAll && currentPreviewIndex >= 0) {
                    List<TimelineComponent.Segment> segs = timelineComponent.getSegments();
                    if (currentPreviewIndex >= segs.size()) {
                        stopPreviewMode();
                        mediaPlayer.controls().pause();
                        return;
                    }
                    TimelineComponent.Segment currentSeg = segs.get(currentPreviewIndex);
                    
                    // If we passed the end of the current segment, jump to next
                    if (newTime >= currentSeg.end) {
                        SwingUtilities.invokeLater(() -> {
                            currentPreviewIndex++;
                            if (currentPreviewIndex < segs.size()) {
                                TimelineComponent.Segment nextSeg = segs.get(currentPreviewIndex);
                                mediaPlayer.controls().setTime(nextSeg.start);
                            } else {
                                mediaPlayer.controls().pause();
                                stopPreviewMode();
                                JOptionPane.showMessageDialog(MainWindow.this, "Preview Complete");
                            }
                        });
                    }
                }
            }

            @Override
            public void finished(MediaPlayer mediaPlayer) {
                SwingUtilities.invokeLater(() -> {
                    btnPlayPause.setText("↺ Replay");
                    long total = mediaPlayer.status().length();
                    timelineComponent.setCurrentTime(total);
                    stopPreviewMode();
                });
            }

            @Override
            public void paused(MediaPlayer mediaPlayer) { SwingUtilities.invokeLater(() -> btnPlayPause.setText("▶ Play")); }
            @Override
            public void playing(MediaPlayer mediaPlayer) { SwingUtilities.invokeLater(() -> btnPlayPause.setText("|| Pause")); }
        });
    }

    // ==========================================
    // Project & File Actions
    // ==========================================

    private void actionNewProject() {
        if (!checkSaveNeeded()) return;
        JFileChooser fileChooser = new JFileChooser();
        fileChooser.setDialogTitle("Select Video for New Project");
        fileChooser.setFileFilter(new FileNameExtensionFilter("Video Files", "mp4", "mkv", "avi", "mov"));

        if (fileChooser.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
            File videoFile = fileChooser.getSelectedFile();
            currentProject = new ProjectModel();
            currentProject.videoPath = videoFile.getAbsolutePath();
            loadProjectIntoUI();
        }
    }

    private void actionOpenProject() {
        if (!checkSaveNeeded()) return;
        JFileChooser fileChooser = new JFileChooser();
        fileChooser.setDialogTitle("Open Project File");
        fileChooser.setFileFilter(new FileNameExtensionFilter("JVSplit Project", "jvs"));

        if (fileChooser.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
            try {
                File file = fileChooser.getSelectedFile();
                currentProject = jsonMapper.readValue(file, ProjectModel.class);
                currentProjectFile = file; 
                loadProjectIntoUI();
            } catch (Exception e) {
                JOptionPane.showMessageDialog(this, "Failed to load: " + e.getMessage());
            }
        }
    }

    /**
     * Saves the current project to JSON.
     * @param saveAs If true, forces the "Save As" dialog even if a file exists.
     */
    private void actionSaveProject(boolean saveAs) {
        if (currentProject == null) return;
        
        // Sync Segments from UI to Model
        currentProject.segments.clear();
        List<TimelineComponent.Segment> uiSegments = timelineComponent.getSegments();
        for (TimelineComponent.Segment uiSeg : uiSegments) {
            currentProject.segments.add(new ProjectModel.Segment(uiSeg.start, uiSeg.end));
        }

        if (currentProjectFile != null && !saveAs) {
            saveToDisk(currentProjectFile);
        } else {
            JFileChooser fileChooser = new JFileChooser();
            fileChooser.setDialogTitle("Save Project");
            fileChooser.setFileFilter(new FileNameExtensionFilter("JVSplit Project", "jvs"));
            
            if (fileChooser.showSaveDialog(this) == JFileChooser.APPROVE_OPTION) {
                File file = fileChooser.getSelectedFile();
                if (!file.getName().endsWith(".json")) file = new File(file.getAbsolutePath() + ".json");
                
                if (file.exists()) {
                    int res = JOptionPane.showConfirmDialog(this, "File exists. Overwrite?", "Confirm", JOptionPane.YES_NO_OPTION);
                    if (res != JOptionPane.YES_OPTION) return;
                }
                saveToDisk(file);
            }
        }
    }
    
    private void saveToDisk(File file) {
        try {
            jsonMapper.writerWithDefaultPrettyPrinter().writeValue(file, currentProject);
            currentProjectFile = file;
            isDirty = false;
            updateTitle();
            if (currentProjectFile == file) System.out.println("Project saved."); 
        } catch (Exception e) {
            JOptionPane.showMessageDialog(this, "Error saving: " + e.getMessage());
        }
    }

    private boolean checkSaveNeeded() {
        if (isDirty && currentProject != null) {
            int result = JOptionPane.showConfirmDialog(this, "You have unsaved changes. Save now?", "Unsaved Changes", JOptionPane.YES_NO_CANCEL_OPTION);
            if (result == JOptionPane.YES_OPTION) {
                actionSaveProject(false);
                return !isDirty; 
            }
            return result == JOptionPane.NO_OPTION;
        }
        return true;
    }

    private void loadProjectIntoUI() {
        if (currentProject == null) return;
        resetProjectState(false); 
        mediaPlayerComponent.mediaPlayer().media().play(currentProject.videoPath);
        mediaPlayerComponent.mediaPlayer().controls().setPause(true);
        isDirty = false;
        updateTitle();
    }

    private void resetProjectState() { resetProjectState(true); }
    private void resetProjectState(boolean fullReset) {
        if (fullReset) {
            currentProject = null;
            currentProjectFile = null;
            isDirty = false;
        }
        if (timelineComponent != null) timelineComponent.clearSegments();
        updateTitle();
        if (fullReset && mediaPlayerComponent != null) mediaPlayerComponent.mediaPlayer().controls().stop();
    }
    
    private void updateTitle() {
        String title = TITLE;
        if (currentProject != null) {
            File f = new File(currentProject.videoPath);
            title += " - " + f.getName();
        }
        if (isDirty) title += " *";
        setTitle(title);
    }
    
    // ==========================================
    // EXPORT LOGIC
    // ==========================================
    
    /**
     * Handles the user interaction for exporting (Destination selection, logic branching).
     * @param singleFile True for merging into one file, False for individual parts.
     */
    private void actionExport(boolean singleFile) {
        if (currentProject == null || timelineComponent.getSegments().isEmpty()) {
            JOptionPane.showMessageDialog(this, "No segments to export!");
            return;
        }
        
        List<TimelineComponent.Segment> segments = timelineComponent.getSegments();
        String inputPath = currentProject.videoPath;
        File sourceFile = new File(inputPath);
        String defaultName = sourceFile.getName().substring(0, sourceFile.getName().lastIndexOf('.'));

        if (singleFile) {
            JFileChooser fileChooser = new JFileChooser();
            fileChooser.setDialogTitle("Save Merged Video");
            fileChooser.setSelectedFile(new File(defaultName + "_merged.mp4"));
            
            if (fileChooser.showSaveDialog(this) == JFileChooser.APPROVE_OPTION) {
                File dest = fileChooser.getSelectedFile();
                if (dest.exists()) {
                    if (JOptionPane.showConfirmDialog(this, "File exists. Overwrite?", "Confirm", JOptionPane.YES_NO_OPTION) != JOptionPane.YES_OPTION) return;
                }
                runExportWorker(true, inputPath, dest, null, segments);
            }
        } else {
            JFileChooser fileChooser = new JFileChooser();
            fileChooser.setDialogTitle("Select Output Folder");
            fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
            
            if (fileChooser.showSaveDialog(this) == JFileChooser.APPROVE_OPTION) {
                File outDir = fileChooser.getSelectedFile();
                
                String baseName = JOptionPane.showInputDialog(this, "Enter base filename for parts:", defaultName);
                if (baseName == null || baseName.trim().isEmpty()) return;
                
                // Pre-flight check for collisions
                boolean collision = false;
                for (int i = 1; i <= segments.size(); i++) {
                    File check = new File(outDir, String.format("%s_part%03d.mp4", baseName, i));
                    if (check.exists()) { collision = true; break; }
                }
                
                if (collision) {
                    int res = JOptionPane.showConfirmDialog(this, "One or more files already exist. Overwrite ALL?", "Confirm Batch Overwrite", JOptionPane.YES_NO_OPTION);
                    if (res != JOptionPane.YES_OPTION) return;
                }
                
                runExportWorker(false, inputPath, outDir, baseName, segments);
            }
        }
    }
    
    /**
     * Creates and returns a modal dialog to block UI during export.
     */
    private JDialog createProcessingDialog() {
        JDialog dialog = new JDialog(this, "Processing", true); // True = Modal
        dialog.setLayout(new BorderLayout());
        
        JPanel content = new JPanel();
        content.setBorder(BorderFactory.createEmptyBorder(20, 20, 20, 20));
        content.setLayout(new BoxLayout(content, BoxLayout.Y_AXIS));
        
        JLabel lblMsg = new JLabel("Exporting video... Please wait.");
        lblMsg.setAlignmentX(JLabel.CENTER_ALIGNMENT);
        
        JProgressBar progressBar = new JProgressBar();
        progressBar.setIndeterminate(true); // Animation
        progressBar.setPreferredSize(new Dimension(250, 20));
        
        content.add(lblMsg);
        content.add(Box.createVerticalStrut(15));
        content.add(progressBar);
        
        dialog.add(content, BorderLayout.CENTER);
        dialog.pack();
        dialog.setLocationRelativeTo(this);
        dialog.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE); // Prevent closing
        
        return dialog;
    }

    /**
     * Executes the FFmpeg export in a background SwingWorker.
     * Shows a modal dialog while processing.
     */
    private void runExportWorker(boolean singleFile, String input, File dest, String baseName, List<TimelineComponent.Segment> segs) {
        JDialog loadingDialog = createProcessingDialog();

        javax.swing.SwingWorker<Void, String> worker = new javax.swing.SwingWorker<Void, String>() {
            @Override
            protected Void doInBackground() throws Exception {
                publish("Starting export...");
                if (singleFile) exportSingleFile(input, dest, segs);
                else exportMultipleFiles(input, dest, baseName, segs);
                return null;
            }
            
            @Override
            protected void process(List<String> chunks) { 
                for (String msg : chunks) System.out.println("[Export] " + msg); 
            }
            
            @Override
            protected void done() {
                // Close the modal dialog to unfreeze UI
                loadingDialog.dispose();
                
                try {
                    get(); // Check for errors
                    int choice = JOptionPane.showConfirmDialog(MainWindow.this, "Export complete! Open location?", "Success", JOptionPane.YES_NO_OPTION);
                    if (choice == JOptionPane.YES_OPTION) openInDefaultPlayer(singleFile ? dest : dest);
                } catch (Exception e) {
                    e.printStackTrace();
                    JOptionPane.showMessageDialog(MainWindow.this, "Export Failed:\n" + e.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
                }
            }
            
            // --- Internal FFmpeg Implementation ---
            private void exportMultipleFiles(String input, File outDir, String baseName, List<TimelineComponent.Segment> segs) throws Exception {
                int index = 1;
                for (TimelineComponent.Segment seg : segs) {
                    String outName = String.format("%s_part%03d.mp4", baseName, index++);
                    File outFile = new File(outDir, outName);
                    publish("Exporting: " + outName);
                    
                    List<String> cmd = new ArrayList<>();
                    cmd.add("ffmpeg"); cmd.add("-y");
                    cmd.add("-i"); cmd.add(input);
                    cmd.add("-ss"); cmd.add(formatTimeFFmpeg(seg.start));
                    cmd.add("-to"); cmd.add(formatTimeFFmpeg(seg.end));
                    cmd.add("-c:v"); cmd.add("libx264");
                    cmd.add("-preset"); cmd.add("ultrafast"); 
                    cmd.add("-c:a"); cmd.add("aac");
                    cmd.add(outFile.getAbsolutePath());
                    runFFmpeg(cmd);
                }
            }

            private void exportSingleFile(String input, File outFile, List<TimelineComponent.Segment> segs) throws Exception {
                File listFile = File.createTempFile("ffmpeg_list", ".txt");
                java.io.PrintWriter writer = new java.io.PrintWriter(listFile);
                String cleanPath = input.replace("\\", "/"); 
                
                for (TimelineComponent.Segment seg : segs) {
                    writer.println("file '" + cleanPath + "'");
                    writer.println("inpoint " + formatTimeFFmpeg(seg.start));
                    writer.println("outpoint " + formatTimeFFmpeg(seg.end));
                }
                writer.close();
                publish("Merging segments...");

                List<String> cmd = new ArrayList<>();
                cmd.add("ffmpeg"); cmd.add("-y");
                cmd.add("-f"); cmd.add("concat");
                cmd.add("-safe"); cmd.add("0");
                cmd.add("-i"); cmd.add(listFile.getAbsolutePath());
                cmd.add("-c:v"); cmd.add("libx264");
                cmd.add("-preset"); cmd.add("ultrafast");
                cmd.add("-c:a"); cmd.add("aac");
                cmd.add(outFile.getAbsolutePath());

                runFFmpeg(cmd);
                listFile.delete(); 
            }

            private void runFFmpeg(List<String> cmd) throws Exception {
                ProcessBuilder pb = new ProcessBuilder(cmd);
                pb.redirectErrorStream(true);
                Process p = pb.start();
                try (java.io.BufferedReader reader = new java.io.BufferedReader(new java.io.InputStreamReader(p.getInputStream()))) {
                    while (reader.readLine() != null) {}
                }
                if (p.waitFor() != 0) throw new RuntimeException("FFmpeg failed");
            }
        };
        
        // Start worker
        worker.execute();
        
        // Block UI until worker calls loadingDialog.dispose()
        loadingDialog.setVisible(true);
    }

    private String formatTimeFFmpeg(long millis) {
        long hours = millis / 3600000;
        long minutes = (millis % 3600000) / 60000;
        long seconds = (millis % 60000) / 1000;
        long ms = millis % 1000;
        return String.format("%02d:%02d:%02d.%03d", hours, minutes, seconds, ms);
    }

    private void openInDefaultPlayer(File file) {
        try { if (java.awt.Desktop.isDesktopSupported()) java.awt.Desktop.getDesktop().open(file); } catch (Exception e) {}
    }
    
    private void showUserGuide() {
        JOptionPane.showMessageDialog(this, "1. File > New Project to open video.\n2. Click 'Mark Selection' or Right-Click timeline.\n3. Right-Click segments to edit/remove.\n4. Export when done.", "Guide", JOptionPane.INFORMATION_MESSAGE);
    }
    private void showAboutDialog() {
        JOptionPane.showMessageDialog(this, "Java Video Splitter v1.0\nPowered by VLCJ & FFmpeg", "About", JOptionPane.INFORMATION_MESSAGE);
    }

    // ==========================================
    // Internal Data Model
    // ==========================================
    public static class ProjectModel {
        public String videoPath;
        public List<Segment> segments = new ArrayList<>();
        public static class Segment {
            public long start; public long end;
            public Segment() {} 
            public Segment(long s, long e) { this.start = s; this.end = e; }
        }
    }
    
    public static void main(String[] args) {
        try { UIManager.setLookAndFeel(new FlatDarkLaf()); } catch (Exception e) {}
        new NativeDiscovery().discover();
        SwingUtilities.invokeLater(() -> new MainWindow().setVisible(true));
    }
}
