package io.github.emmrida.jvsplit;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Random;
import java.util.function.Consumer;

import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;
import javax.swing.SwingUtilities;

public class TimelineComponent extends JComponent {

    // --- Data Model ---
    private long totalDurationMs = 1;
    private long currentTimeMs = 0;
    
    public static class Segment {
        public long start;
        public long end;
        public Color color; 

        public Segment(long s, long e) { 
            this.start = s; 
            this.end = e;
            float hue = new Random().nextFloat();
            this.color = Color.getHSBColor(hue, 0.6f, 0.9f);
        }
    }
    
    private List<Segment> segments = new ArrayList<>();

    // --- Callbacks ---
    private Consumer<Long> seekCallback;       
    private Consumer<Segment> playRangeCallback; 
    private Runnable playAllCallback;          
    private Runnable markCurrentCallback;      

    // --- Visual Constants ---
    private static final int HEIGHT = 80; 
    private static final int RULER_H = 20;
    private static final int TRACK_H = 40;
    private static final int TRACK_Y = 30; 

    public TimelineComponent(
            Consumer<Long> seekCallback, 
            Consumer<Segment> playRangeCallback,
            Runnable playAllCallback,
            Runnable markCurrentCallback) {
        
        this.seekCallback = seekCallback;
        this.playRangeCallback = playRangeCallback;
        this.playAllCallback = playAllCallback;
        this.markCurrentCallback = markCurrentCallback;
        
        setPreferredSize(new Dimension(800, HEIGHT));
        setFont(new Font("SansSerif", Font.PLAIN, 10));
        
        MouseAdapter mouseHandler = new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                if (e.isPopupTrigger()) handlePopup(e);
                else if (SwingUtilities.isLeftMouseButton(e)) updateTimeFromMouse(e.getX());
            }

            @Override
            public void mouseReleased(MouseEvent e) {
                if (e.isPopupTrigger()) handlePopup(e);
            }

            @Override
            public void mouseDragged(MouseEvent e) {
                if (!e.isPopupTrigger() && SwingUtilities.isLeftMouseButton(e)) {
                    updateTimeFromMouse(e.getX());
                }
            }
        };
        
        addMouseListener(mouseHandler);
        addMouseMotionListener(mouseHandler);
    }

    // --- Core API ---
    public void setTotalDuration(long duration) {
        this.totalDurationMs = duration > 0 ? duration : 1;
        repaint();
    }
    public void setCurrentTime(long time) { this.currentTimeMs = time; repaint(); }
    public List<Segment> getSegments() { return segments; }

    public boolean isOverlapping(long start, long end, Segment excludeSegment) {
        for (Segment s : segments) {
            if (s == excludeSegment) continue; 
            if (start < s.end && end > s.start) return true;
        }
        return false;
    }
    
    public boolean isOverlapping(long start, long end) {
        return isOverlapping(start, end, null);
    }

    public void addSegment(long start, long end) {
        start = Math.max(0, start);
        end = Math.min(totalDurationMs, end);
        if (start < end) {
            segments.add(new Segment(start, end));
            Collections.sort(segments, Comparator.comparingLong(s -> s.start));
            repaint();
        }
    }
    
    public void clearSegments() { segments.clear(); repaint(); }

    // --- Logic ---
    private void updateTimeFromMouse(int mouseX) {
        double percentage = (double) Math.max(0, Math.min(getWidth(), mouseX)) / getWidth();
        long newTime = (long) (percentage * totalDurationMs);
        this.currentTimeMs = newTime;
        repaint();
        if (seekCallback != null) seekCallback.accept(newTime);
    }

    private Segment getSegmentAt(int x) {
        double pxPerMs = (double) getWidth() / totalDurationMs;
        for (int i = segments.size() - 1; i >= 0; i--) {
            Segment s = segments.get(i);
            int startX = (int) (s.start * pxPerMs);
            int endX = (int) (s.end * pxPerMs);
            if (x >= startX - 2 && x <= endX + 2) return s;
        }
        return null;
    }

    // --- Popup Logic ---
    private void handlePopup(MouseEvent e) {
        Segment target = getSegmentAt(e.getX());
        if (target != null) showSegmentPopup(e, target);
        else showBackgroundPopup(e);
    }

    private void showSegmentPopup(MouseEvent e, Segment segment) {
        JPopupMenu menu = new JPopupMenu();
        
        JMenuItem itemStart = new JMenuItem("Set selection start");
        itemStart.addActionListener(ev -> {
            if (currentTimeMs < segment.end && !isOverlapping(currentTimeMs, segment.end, segment)) { 
                segment.start = currentTimeMs; repaint(); 
            } else java.awt.Toolkit.getDefaultToolkit().beep();
        });
        
        JMenuItem itemEnd = new JMenuItem("Set selection end");
        itemEnd.addActionListener(ev -> {
            if (currentTimeMs > segment.start && !isOverlapping(segment.start, currentTimeMs, segment)) { 
                segment.end = currentTimeMs; repaint(); 
            } else java.awt.Toolkit.getDefaultToolkit().beep();
        });

        JMenuItem itemPlay = new JMenuItem("▶ Play this selection");
        itemPlay.addActionListener(ev -> { if (playRangeCallback != null) playRangeCallback.accept(segment); });

        JMenuItem itemRemove = new JMenuItem("🗑 Remove");
        itemRemove.setForeground(Color.RED);
        itemRemove.addActionListener(ev -> { 
            // FINALIZATION: Confirm Removal
            int choice = JOptionPane.showConfirmDialog(this, 
                "Are you sure you want to remove this segment?", 
                "Confirm Removal", JOptionPane.YES_NO_OPTION);
            
            if (choice == JOptionPane.YES_OPTION) {
                segments.remove(segment); 
                repaint(); 
            }
        });

        menu.add(itemStart);
        menu.add(itemEnd);
        menu.add(new JSeparator());
        menu.add(itemPlay);
        menu.add(new JSeparator());
        menu.add(itemRemove);
        
        menu.show(this, e.getX(), e.getY());
    }

    private void showBackgroundPopup(MouseEvent e) {
        JPopupMenu menu = new JPopupMenu();
        JMenuItem itemPlayAll = new JMenuItem("▶▶ Play all selections");
        itemPlayAll.addActionListener(ev -> { if (playAllCallback != null) playAllCallback.run(); });
        
        JMenuItem itemMark = new JMenuItem("✂ Mark selection here");
        itemMark.addActionListener(ev -> {
            updateTimeFromMouse(e.getX()); 
            if (markCurrentCallback != null) markCurrentCallback.run();
        });

        menu.add(itemPlayAll);
        menu.add(new JSeparator());
        menu.add(itemMark);
        menu.show(this, e.getX(), e.getY());
    }

    // --- Drawing Logic ---
    @Override
    protected void paintComponent(Graphics g) {
        super.paintComponent(g);
        Graphics2D g2 = (Graphics2D) g;
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

        int w = getWidth();
        double pxPerMs = (double) w / totalDurationMs;

        drawRuler(g2, w, pxPerMs);

        g2.setColor(new Color(40, 40, 40));
        g2.fillRoundRect(0, TRACK_Y, w, TRACK_H, 8, 8);

        for (Segment s : segments) {
            int x1 = (int) (s.start * pxPerMs);
            int x2 = (int) (s.end * pxPerMs);
            int segW = Math.max(3, x2 - x1);
            
            g2.setColor(s.color);
            g2.fillRect(x1, TRACK_Y, segW, TRACK_H);
            g2.setColor(s.color.brighter());
            g2.drawRect(x1, TRACK_Y, segW, TRACK_H);
        }

        int playX = (int) (currentTimeMs * pxPerMs);
        g2.setColor(new Color(255, 69, 58)); 
        g2.setStroke(new BasicStroke(2));
        g2.drawLine(playX, 0, playX, getHeight());
        g2.fillPolygon(new int[]{playX-5, playX+5, playX}, new int[]{RULER_H, RULER_H, RULER_H+10}, 3);
    }

    private void drawRuler(Graphics2D g2, int width, double pxPerMs) {
        g2.setColor(Color.GRAY);
        g2.drawLine(0, RULER_H, width, RULER_H); 

        double minPxBetweenTicks = 80.0;
        long minMsInterval = (long) (minPxBetweenTicks / pxPerMs);
        long intervalMs = getNicelyRoundedInterval(minMsInterval);

        for (long t = 0; t <= totalDurationMs; t += intervalMs) {
            int x = (int) (t * pxPerMs);
            if (x > width) break; 

            g2.drawLine(x, 0, x, RULER_H);
            String timeStr = formatTimeShort(t);
            g2.drawString(timeStr, x + 4, RULER_H - 5);
        }
    }

    private long getNicelyRoundedInterval(long minMs) {
        long[] steps = {
            1000, 2000, 5000, 10000, 30000,       
            60000, 120000, 300000, 600000,        
            1800000, 3600000                      
        };
        for (long step : steps) {
            if (step >= minMs) return step;
        }
        return 3600000; 
    }

    private String formatTimeShort(long ms) {
        long sec = ms / 1000;
        if (sec < 60) return sec + "s";
        long min = sec / 60;
        return min + "m";
    }
}
