package chatclient;

import java.awt.Component;
import java.awt.ComponentOrientation;
import java.awt.Container;
import java.awt.Image;
import java.awt.Window;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JEditorPane;
import javax.swing.text.html.HTMLEditorKit;

/**
 *
 */
public class Helper {
	private static final String ALNUM_CHARS = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";

	/**
	 * Get the root window of a given component.
	 * @param component Component to get the root window of.
	 * @return Root window of the component.
	 */
    public static Window getRootWindow(Component component) {
        if (component == null) {
            return null;
        }
        
        // Traverse up the component hierarchy
        Container parent = component.getParent();
        Window rootWindow = null;
        
        // If the component itself is a Window, return it
        if (component instanceof Window) {
            return (Window) component;
        }
        
        // Traverse up the hierarchy to find the Window ancestor
        while (parent != null) {
            if (parent instanceof Window) {
                rootWindow = (Window) parent;
                break;
            }
            parent = parent.getParent();
        }
        
        return rootWindow;
    }

	/**
	 * Gets a random string
	 * @param length The length
	 * @return The random string
	 */
	public static String getRandomString(int length) {
		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < length; i++)
			sb.append(ALNUM_CHARS.charAt((int) (Math.random() * ALNUM_CHARS.length())));
		return sb.toString();
	}

	/**
	 * Sets up the editor pane
	 * @param editorPane The editor pane
	 */
	public static void setupEditorPane(JEditorPane editorPane) {
		editorPane.setContentType("text/html"); //$NON-NLS-1$
		editorPane.setEditorKit(new HTMLEditorKit());
	}

	/**
	 * Gets the local time
	 * @return The local time
	 */
	public static String getLocalTime() {
		SimpleDateFormat sdf = new SimpleDateFormat("HH:mm"); //$NON-NLS-1$
		return sdf.format(new Date());
	}

	/**
	 * Gets the local time
	 * @param time The time
	 * @return The local time
	 */
	public static String getLocalTime(long time) {
		SimpleDateFormat sdf = new SimpleDateFormat("HH:mm"); //$NON-NLS-1$
		return sdf.format(new Date(time));
	}

	/**
	 * Loads an icon from resources and resizes it
	 * @param resourcePath The path to the icon in resources folder
	 * @param width The width of the icon to return
	 * @param height The height of the icon to return
	 * @return The icon
	 */
	public static Icon loadIconFromResources(String resourcePath, int width, int height) {
		try {
			URL resourceURL = Helper.class.getResource(resourcePath);
			if (resourceURL == null)
				throw new IllegalArgumentException("Resource not found: " + resourcePath); //$NON-NLS-1$
			ImageIcon imageIcon = new ImageIcon(resourceURL);
			Image originalImage = imageIcon.getImage();
			Image resizedImage = originalImage.getScaledInstance(width, height, Image.SCALE_SMOOTH);
			return new ImageIcon(resizedImage);
		} catch (Exception e) {
			System.err.println("Error loading icon: " + resourcePath); //$NON-NLS-1$
		}
		return null;
	}

	/**
	 * Checks if current locale is RTL
	 * @param locale The locale to check
	 * @return True if RTL
	 */
	public static boolean isRTL(Locale locale) {
		char firstChar = locale.getDisplayName(locale).charAt(0);
		byte directionality = Character.getDirectionality(firstChar);
		return directionality == Character.DIRECTIONALITY_RIGHT_TO_LEFT || directionality == Character.DIRECTIONALITY_RIGHT_TO_LEFT_ARABIC;
	}

	/**
	 * Enables RTL when needed. Depends on current locale.
	 * @param container Parent component to enable RTL and its children
	 */
	public static void enableRtlWhenNeeded(Container container) {
		if(!isRTL(Locale.getDefault()))
			return;
		container.applyComponentOrientation(ComponentOrientation.RIGHT_TO_LEFT);
		for (Component component : container.getComponents())
			if (component instanceof Container)
				enableRtlWhenNeeded((Container) component);
		container.revalidate();
		container.repaint();
	}

	/**
	 *
	 * @param text
	 * @return
	 */
	public static String escapeHtml(String text) {
	    if (text == null)
	        return null;

	    StringBuffer sb = new StringBuffer(text.length() * 2);
	    for (int i = 0; i < text.length(); i++) {
	        char c = text.charAt(i);
	        if (c == '&') {
	            sb.append("&amp;");
	        } else if (c == '<') {
	            sb.append("&lt;");
	        } else if (c == '>') {
	            sb.append("&gt;");
	        } else if (c == '"') {
	            sb.append("&quot;");
	        } else if (c == '\'') {
	            sb.append("&#39;");
	        } else sb.append(c);
	    }
	    return sb.toString();
	}
}












