/**
 * Copyright (c) 2025 El Mhadder Mohamed Rida. All rights reserved.
 * This code is licensed under the [MIT License](https://opensource.org/licenses/MIT).
 */
package chatclient;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.border.EmptyBorder;
import java.awt.Toolkit;

import javax.swing.GroupLayout;
import javax.swing.GroupLayout.Alignment;
import javax.swing.BoxLayout;
import java.awt.BorderLayout;
import javax.swing.JToolBar;
import javax.swing.JScrollPane;
import javax.swing.JButton;
import javax.swing.JTextField;
import javax.swing.border.EtchedBorder;

import chatclient.ServerConnector.ApiResponse;

import javax.swing.ScrollPaneConstants;
import javax.swing.SwingUtilities;
import javax.swing.Timer;
import javax.swing.JToggleButton;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import javax.swing.JProgressBar;
import javax.swing.JScrollBar;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import javax.swing.UIManager;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.Component;
import javax.swing.Box;

import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

/**
 * The main frame of the internal chat client
 *
 * @author El Mhadder Mohamed Rida
 */
public class MainFrame extends JFrame {
	private static final long serialVersionUID = 1L;

	private static enum UISTATE { LOGOUT, DISCONNECTED, CONNECTED, WAITING, IDLE };

	private ServerConnector serverConnector = null;

	private boolean isChatbotWaiting = false;

	private JPanel contentPane;
	private JTextField tfUserMsg;
	private JPanel pnlChat;
	private JButton btnSend;
	private JToggleButton tglConnect;
	private JScrollPane spChat;
	private JProgressBar pbWait;
	private Component horizontalStrut;
	private Component horizontalStrut_1;
	private JTextField tfUserName;

	/**
	 * Called when a remote message link is clicked by the user
	 * @param id ID of the message
	 */
	public void onRemoteMsgLinkClicked(String id) {
		if(id != null && !id.isBlank()) {
			int us = id.indexOf('_'); // TODO : Document this feature.
			if(us >= 0) {
				String response = id.substring(us).replaceAll("_", " "); //$NON-NLS-2$
				tfUserMsg.setText(response);
				btnSend.doClick();
			}
		}
	}

	/**
	 * Sets the UI mode
	 * @param mode The UI mode
	 */
	private void setUIMode(UISTATE mode) {
		pbWait.setEnabled(false);
		tfUserName.setEditable(false);
		tfUserMsg.setEnabled(false);
		btnSend.setEnabled(false);
		pbWait.setVisible(false);
		switch(mode) {
		case LOGOUT:
			tglConnect.setSelected(false);
			tfUserName.requestFocus();
			break;
		case DISCONNECTED:
			tglConnect.setSelected(false);
			tfUserName.setEditable(true);
			tfUserName.requestFocus();
			break;
		case CONNECTED:
			pnlChat.removeAll();
			btnSend.setEnabled(true);
			tfUserMsg.setEnabled(true);
			break;
		case WAITING:
			pbWait.setVisible(true);
			tfUserMsg.setText("");
			break;
		case IDLE:
			btnSend.setEnabled(true);
			tfUserMsg.setEnabled(true);
			tfUserMsg.requestFocus();
			break;
		}
	}

	/**
	 * Scrolls the chat window to the bottom
	 */
	private void scrollToBottom() {
		Timer timer = new Timer(10, null);
		ActionListener al = new ActionListener() {
			float steps = -1.0f;
			@Override
			public void actionPerformed(ActionEvent e) {
				JScrollBar bar = spChat.getVerticalScrollBar();
				int max = bar.getMaximum() - bar.getVisibleAmount();
				if(steps < 0.0f)
					steps = (max - bar.getValue()) / 10;
				bar.setValue(bar.getValue() + Math.round(steps*=0.915f));
				if(steps <= 1.0f)
					steps = 1.0f;
				if(bar.getValue() >= max)
					timer.stop();
			}
		};
		timer.addActionListener(al);
		timer.start();
	}

	/**
	 * Create the frame and initialize
	 */
	public MainFrame() {
		addWindowListener(new WindowAdapter() {
			@Override
			public void windowOpened(WindowEvent e) {
				ConnectDialog dlg = new ConnectDialog();
				dlg.setLocationRelativeTo(MainFrame.this);
				dlg.setVisible(true);
				if(!dlg.isCancelled()) {
					pbWait.setVisible(true);
	                serverConnector = new ServerConnector(dlg.getKey1(), dlg.getKey2(), "https://" + dlg.getHost() + ":" + dlg.getPort());
					new Thread(() -> {
	                    if(serverConnector.login()) {
	                    	SwingUtilities.invokeLater(() -> {
	                    		setUIMode(UISTATE.DISCONNECTED);
	                    	});
	                    } else {
	                    	SwingUtilities.invokeLater(() -> {
		                    	JOptionPane.showMessageDialog(MainFrame.this, "Login failed", "Error", JOptionPane.ERROR_MESSAGE);
		                    	dispatchEvent(new WindowEvent(MainFrame.this, WindowEvent.WINDOW_CLOSING));
	                    	});
	                    }
					}).start();
				} else System.exit(0);
			}
			@Override
			public void windowClosing(WindowEvent e) {
				if(!serverConnector.logout())
					JOptionPane.showMessageDialog(MainFrame.this, "Logout failed", "Error", JOptionPane.ERROR_MESSAGE);
				serverConnector = null;
				dispose();
				System.exit(0);
			}
		});
		setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
		setSize(400, 600);
		setLocationRelativeTo(null);
		setTitle("Chat4Us-Client - Sample Project");
		contentPane = new JPanel();
		contentPane.setBorder(new EmptyBorder(5, 5, 5, 5));
		setContentPane(contentPane);
		contentPane.setLayout(new BorderLayout(0, 0));

		JToolBar toolBar = new JToolBar();
		toolBar.setFloatable(false);
		toolBar.setBorder(new EtchedBorder(EtchedBorder.LOWERED, null, null));
		contentPane.add(toolBar, BorderLayout.NORTH);

		tglConnect = new JToggleButton("");
		tglConnect.addItemListener(e -> {
			if(serverConnector.isLoggedIn()) {
				if(tglConnect.isSelected()) {
					if(tfUserName.getText().isEmpty()) {
						tglConnect.setSelected(false);
						tfUserName.requestFocus();
						JOptionPane.showMessageDialog(MainFrame.this, "Enter user name", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}
					pnlChat.removeAll();
					new Thread(() -> {
						ApiResponse response = serverConnector.letsChat(tfUserName.getText());
						if(response != null && response.isSuccess()) {
							String[] messages = response.getMessages();
							SwingUtilities.invokeLater(() -> {
								setUIMode(UISTATE.CONNECTED);
								IMessagePanel panel;
								for(String message : messages) {
									panel = new RemoteMessage();
									panel.setMessage(message, System.currentTimeMillis());
									pnlChat.add((JPanel)panel);
									pnlChat.revalidate();
									pnlChat.repaint();
								}
								if(!response.isChatEnded()) {
									setUIMode(UISTATE.IDLE);
								}
							});
						} else {
							SwingUtilities.invokeLater(() -> {
								setUIMode(UISTATE.LOGOUT);
		                        JOptionPane.showMessageDialog(MainFrame.this, "Chat start failure", "Error", JOptionPane.ERROR_MESSAGE);
							});
						}
					}).start();
				} else {
					SwingUtilities.invokeLater(() -> setUIMode(UISTATE.DISCONNECTED));
				}
			} else JOptionPane.showMessageDialog(MainFrame.this, "User not logged in!", "Error", JOptionPane.ERROR_MESSAGE);
		});
		tglConnect.setIcon(Helper.loadIconFromResources("/disconnected.png", 20, 20));
		tglConnect.setSelectedIcon(Helper.loadIconFromResources("/connected.png", 20, 20));
		tglConnect.setFocusable(false);
		toolBar.add(tglConnect);

		tfUserName = new JTextField();
		tfUserName.addKeyListener(new KeyAdapter() {
			@Override
			public void keyReleased(KeyEvent e) {
				if(e.getKeyCode() == KeyEvent.VK_ENTER)
					tglConnect.doClick();
			}
		});
		tfUserName.setEditable(false);
		tfUserName.setColumns(10);
		tfUserName.setMaximumSize(new Dimension(128, 20));
		tfUserName.setBorder(new EmptyBorder(0, 3, 0, 3));
		toolBar.add(tfUserName);

		pbWait = new JProgressBar();
		pbWait.setVisible(false);
		pbWait.setBorder(null);
		pbWait.setMaximumSize(new Dimension(32767, 20));

		horizontalStrut = Box.createHorizontalStrut(20);
		horizontalStrut.setPreferredSize(new Dimension(5, 0));
		horizontalStrut.setMinimumSize(new Dimension(5, 0));
		toolBar.add(horizontalStrut);
		pbWait.setIndeterminate(true);
		toolBar.add(pbWait);

		horizontalStrut_1 = Box.createHorizontalStrut(20);
		horizontalStrut_1.setPreferredSize(new Dimension(5, 0));
		horizontalStrut_1.setMinimumSize(new Dimension(5, 0));
		toolBar.add(horizontalStrut_1);

		JPanel panel = new JPanel();
		panel.setBorder(new EtchedBorder(EtchedBorder.LOWERED, null, null));
		contentPane.add(panel, BorderLayout.SOUTH);

		tfUserMsg = new JTextField();
		tfUserMsg.addKeyListener(new KeyAdapter() {
			@Override
			public void keyReleased(KeyEvent e) {
				if(e.getKeyCode() == KeyEvent.VK_ENTER) {
					btnSend.doClick();
				}
			}
		});
		tfUserMsg.setEnabled(false);
		tfUserMsg.setColumns(10);

		btnSend = new JButton();
		btnSend.setIcon(Helper.loadIconFromResources("/send.png", 20, 20));
		btnSend.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				if(serverConnector != null && serverConnector.isLoggedIn()) {
					String msg = tfUserMsg.getText();
					if(!msg.isBlank()) {
						setUIMode(UISTATE.WAITING);
						if(!msg.equals("...") && !isChatbotWaiting) {
							IMessagePanel panel = new UserMessage();
							panel.setMessage(Helper.escapeHtml(msg), System.currentTimeMillis());
							pnlChat.add((JPanel)panel);
							scrollToBottom();
							pnlChat.revalidate();
							pnlChat.repaint();
						}
						new Thread(() -> {
							ApiResponse response = serverConnector.sendMessage(tfUserName.getText(), msg);
							if(response != null) {
								handleServerAnswer(response);
							} else { // Fatal error or disconnected
								SwingUtilities.invokeLater(() -> {
									setUIMode(UISTATE.DISCONNECTED);
									appendErrorMessage("Fatal error or disconnected");
								});
							}
						}).start();
					} else Toolkit.getDefaultToolkit().beep();
				} else {
					setUIMode(UISTATE.DISCONNECTED);
					JOptionPane.showMessageDialog(MainFrame.this, "Your are not logged in. Please close this window and try again.", "Error", JOptionPane.ERROR_MESSAGE); //$NON-NLS-2$
				}
			}

			/**
			 *
			 * @param response
			 */
			private void handleServerAnswer(ApiResponse response) {
				if(response != null && response.isSuccess()) {
					String[] messages = response.getMessages();
					for(String message : messages) {
						if(!message.isBlank()) {
							SwingUtilities.invokeLater(() -> {
								IMessagePanel pnl;
								pnl = new RemoteMessage();
								pnl.setMessage(message, System.currentTimeMillis());
								pnl.setIcon("/" + response.getChatStateIconName() + ".png"); //$NON-NLS-2$
								pnlChat.add((JPanel)pnl);
							});
						}
					}
					if(!response.isChatEnded()) {
						SwingUtilities.invokeLater(() -> {
							setUIMode(UISTATE.IDLE);
							scrollToBottom();
						});
					} else {
						SwingUtilities.invokeLater(() -> {
							setUIMode(UISTATE.DISCONNECTED);
                            appendErrorMessage("This chat has ended.");
							scrollToBottom();
                        });
					}
					isChatbotWaiting = response.isChatbotWaiting();
					if(isChatbotWaiting) {
						SwingUtilities.invokeLater(() -> {
							// Should send "..." back to server in order to connect to an agent/messenger.
							tfUserMsg.setText("...");
							btnSend.doClick();
						});
					}
				} else if(response != null) {
					SwingUtilities.invokeLater(() -> {
						appendErrorMessage("Error sending message.");
					});
				}
				SwingUtilities.invokeLater(() -> {
					pnlChat.revalidate();
					pnlChat.repaint();
				});
			}

			/**
			 *
			 * @param msg
			 */
			public void appendErrorMessage(String msg) {
				IMessagePanel pnl = new ErrorMessage();
				pnl.setMessage(msg, System.currentTimeMillis());
				pnlChat.add((JPanel)pnl);
			}
		});
		btnSend.setEnabled(false);
		GroupLayout gl_panel = new GroupLayout(panel);
		gl_panel.setHorizontalGroup(
			gl_panel.createParallelGroup(Alignment.TRAILING)
				.addGroup(gl_panel.createSequentialGroup()
					.addGap(1)
					.addComponent(tfUserMsg, GroupLayout.DEFAULT_SIZE, 363, Short.MAX_VALUE)
					.addGap(3)
					.addComponent(btnSend)
					.addGap(1))
		);
		gl_panel.setVerticalGroup(
			gl_panel.createParallelGroup(Alignment.LEADING)
				.addGroup(gl_panel.createSequentialGroup()
					.addGroup(gl_panel.createParallelGroup(Alignment.BASELINE)
						.addComponent(tfUserMsg, GroupLayout.DEFAULT_SIZE, 38, Short.MAX_VALUE)
						.addComponent(btnSend, GroupLayout.DEFAULT_SIZE, GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
					.addGap(1))
		);
		panel.setLayout(gl_panel);

		spChat = new JScrollPane();
		spChat.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
		spChat.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
		contentPane.add(spChat, BorderLayout.CENTER);

		pnlChat = new JPanel();
		pnlChat.setBackground(UIManager.getColor("window"));
		spChat.setViewportView(pnlChat);
		pnlChat.setLayout(new BoxLayout(pnlChat, BoxLayout.Y_AXIS));

	}
	
	/**
	 * Launch the application.
	 */
	public static void main(String[] args) {
		EventQueue.invokeLater(new Runnable() {
			public void run() {
				try {
					MainFrame frame = new MainFrame();
					frame.setVisible(true);
				} catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		});
	}
}
