package chatclient;

import java.net.URI;
import java.net.URLEncoder;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpRequest.BodyPublishers;
import java.net.http.HttpResponse.BodyHandlers;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Arrays;
import java.util.Map;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLParameters;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.annotations.SerializedName;

/**
 * Manages the connection to the bot server
 */
public class ServerConnector {

	private static final int API_TIMEOUT = 120; // Server API timeout in seconds

    private final String key1; // First authentication key
    private final String key2; // Second authentication key
    private final String botUrl; //URL of the bot server
    private final HttpClient httpClient; //HTTP client for making requests

    private String serverToken; // Server token returned after authentication

    /**
     * Constructor for ServerConnector
     * @param key1 First authentication key
     * @param key2 Second authentication key
     * @param botUrl Base URL of the bot server
     */
    public ServerConnector(String key1, String key2, String botUrl) {
        this.key1 = key1;
        this.key2 = key2;
        this.botUrl = normalizeUrl(botUrl);
        this.httpClient = createHttpClient();
        this.serverToken = null;
    }

    /**
     * Normalizes the URL by ensuring it doesn't end with a slash
     * @param url The URL to normalize
     * @return The normalized URL
     */
    private String normalizeUrl(String url) {
        if (url == null || url.trim().isEmpty()) {
            throw new IllegalArgumentException("URL cannot be null or empty");
        }
        return url.endsWith("/") ? url.substring(0, url.length() - 1) : url;
    }

    /**
     * Creates an HTTP client that is used to communicate with the bot server
     * @return The HTTP client
     */
    private HttpClient createHttpClient() {
        try {
        	/*
        	 * Added code to bypass SSL verification to allow secure communication
        	 * with the bot server using self-signed certificates.
        	 */
        	/* Disable on production */
            TrustManager[] trustAllCerts = new TrustManager[] {
                new X509TrustManager() {
    				public void checkClientTrusted(X509Certificate[] chain, String authType) throws CertificateException { }
    				public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException { }
    				public X509Certificate[] getAcceptedIssuers() { return new X509Certificate[0]; }
                }
            };
            /**/

            //sslContext.init(null, null, null);

            /* Disable on production */
            SSLContext sslContext = SSLContext.getInstance("TLS");
			sslContext.init(null, trustAllCerts, new java.security.SecureRandom());
			/**/

			/* Disable on production */
            SSLContext.setDefault(sslContext);
            SSLParameters sslParameters = new SSLParameters();
            sslParameters.setEndpointIdentificationAlgorithm(null);
            HostnameVerifier allHostsValid = (hostname, session) -> true;
            HttpsURLConnection.setDefaultHostnameVerifier(allHostsValid);
            HttpsURLConnection.setDefaultSSLSocketFactory(sslContext.getSocketFactory());
            /**/

            return HttpClient.newBuilder()
                    .sslContext(sslContext)
                    .sslParameters(sslParameters)
                    .connectTimeout(Duration.ofSeconds(API_TIMEOUT))
                    .version(HttpClient.Version.HTTP_1_1)
                    .build();

        } catch (Exception e) {
            throw new RuntimeException("Failed to create HTTP client with SSL bypass", e);
        }
    }

    /**
     * Performs login operation and stores the token
     * @return true if login successful, false otherwise
     */
    public boolean login() {
        if (serverToken != null) {
            System.err.println("Already logged in!");
            return true;
        }

        try {
        	// Credentials to login
            Map<String, String> formData = Map.of(
                "key1", key1,
                "key2", key2
            );

            HttpRequest request = HttpRequest.newBuilder()
                    .uri(URI.create(botUrl + "/login"))
                    .header("Content-Type", "application/x-www-form-urlencoded")
                    .POST(BodyPublishers.ofString(buildQueryString(formData)))
                    .timeout(Duration.ofSeconds(API_TIMEOUT))
                    .build();

            HttpResponse<String> response = httpClient.send(request, BodyHandlers.ofString());

            ApiResponse apiResponse = parseResponseWithAdditionalFields(response.body());

            if (apiResponse != null && apiResponse.isSuccess() && apiResponse.getToken() != null) {
                serverToken = apiResponse.getToken();
                return true;
            } else {
                System.err.println("Login failed. Status: " +
                    (apiResponse != null ? apiResponse.isSuccess() : "null"));
                serverToken = null;
                return false;
            }
        } catch (Exception e) {
            System.err.println("Login error: " + e.getMessage());
            e.printStackTrace();
            return false;
        }
    }

    /**
     * Performs logout operation and removes the token
     * @return true if logout successful or no token was present, false on error
     */
    public boolean logout() {
        if (serverToken == null) {
            return true; // No token to logout, consider it successful
        }

        try {
            Map<String, String> formData = Map.of("token", serverToken);

            HttpRequest request = HttpRequest.newBuilder()
                    .uri(URI.create(botUrl + "/logout"))
                    .header("Content-Type", "application/x-www-form-urlencoded")
                    .POST(BodyPublishers.ofString(buildQueryString(formData)))
                    .timeout(Duration.ofSeconds(API_TIMEOUT))
                    .build();

            HttpResponse<String> response = httpClient.send(request, BodyHandlers.ofString());
            // TODO : check response
            
            serverToken = null;
            httpClient.close();
            return true;
        } catch (Exception e) {
            System.err.println("Logout error: " + e.getMessage());
            serverToken = null;
            return false;
        }
    }

    /**
     * Initiates a chat session
     * @param userId The user ID to start chat with
     * @return ApiResponse if successful, null otherwise
     */
    public ApiResponse letsChat(String userId) {
        if (serverToken == null) {
            return null;
        }

        try {
            Map<String, String> formData = Map.of(
                "usr_id", userId,
                "token", serverToken
            );

            HttpRequest request = HttpRequest.newBuilder()
                    .uri(URI.create(botUrl + "/letschat"))
                    .header("Content-Type", "application/x-www-form-urlencoded")
                    .POST(BodyPublishers.ofString(buildQueryString(formData)))
                    .timeout(Duration.ofSeconds(API_TIMEOUT))
                    .build();

            HttpResponse<String> response = httpClient.send(request, BodyHandlers.ofString());
            ApiResponse apiResponse = parseResponseWithAdditionalFields(response.body());

            return (apiResponse != null && apiResponse.isSuccess()) ? apiResponse : null;
        } catch (Exception e) {
            System.err.println("LetsChat error: " + e.getMessage());
            return null;
        }
    }

    /**
     * Sends a message to the chat bot server
     * @param userId The ID of the user sending the message
     * @param message The message content
     * @return ApiResponse if successful, null otherwise
     */
    public ApiResponse sendMessage(String userId, String message) {
        if (serverToken == null) {
            return null;
        }

        try {
            Map<String, String> formData = Map.of(
                "usr_id", userId,
                "token", serverToken,
                "message", message
            );

            HttpRequest request = HttpRequest.newBuilder()
                    .uri(URI.create(botUrl + "/message"))
                    .header("Content-Type", "application/x-www-form-urlencoded")
                    .POST(BodyPublishers.ofString(buildQueryString(formData)))
                    .timeout(Duration.ofSeconds(API_TIMEOUT))
                    .build();

            HttpResponse<String> response = httpClient.send(request, BodyHandlers.ofString());
            ApiResponse apiResponse = parseResponseWithAdditionalFields(response.body());

            return (apiResponse != null && apiResponse.isSuccess()) ? apiResponse : null;
        } catch (Exception e) {
            System.err.println("SendMessage error: " + e.getMessage());
            return null;
        }
    }

    /**
     * Helper method to build URL-encoded query string
     * @param parameters Query parameters
     */
    private String buildQueryString(Map<String, String> parameters) {
        StringBuilder result = new StringBuilder();
        for (Map.Entry<String, String> entry : parameters.entrySet()) {
            if (result.length() > 0) {
                result.append("&");
            }
            result.append(URLEncoder.encode(entry.getKey(), StandardCharsets.UTF_8))
                  .append("=")
                  .append(URLEncoder.encode(entry.getValue(), StandardCharsets.UTF_8));
        }
        return result.toString();
    }

    /**
     * Helper method to parse JSON response with additional fields.
     * See ApiResponse class for more details...
     * @param jsonResponse The JSON response string
     * @return ApiResponse if successful, null otherwise
     */
    private ApiResponse parseResponseWithAdditionalFields(String jsonResponse) {
        if (jsonResponse == null || jsonResponse.trim().isEmpty()) {
            return null;
        }

        try {
            JsonObject jsonObject = JsonParser.parseString(jsonResponse).getAsJsonObject();
            String status = jsonObject.has("STATUS") ? jsonObject.get("STATUS").getAsString() : null;
			String token = jsonObject.has("TOKEN") ? jsonObject.get("TOKEN").getAsString() : null;
			boolean chatEnded = jsonObject.has("CHAT_ENDED") ? jsonObject.get("CHAT_ENDED").getAsBoolean() : false;
			String chatState = jsonObject.has("CHAT_STATE") ? jsonObject.get("CHAT_STATE").getAsString() : null;
			String locale = jsonObject.has("LOCALE") ? jsonObject.get("LOCALE").getAsString() : null;
			boolean chatbotWaiting = jsonObject.has("CHATBOT_WAITING") ? jsonObject.get("CHATBOT_WAITING").getAsBoolean() : false;
			String[] messages = null;
			if(jsonObject.has("CHATBOT_MESSAGE")) {
				JsonArray ja = jsonObject.get("CHATBOT_MESSAGE").getAsJsonArray();
				messages = new String[ja.size()];
				for(int i = 0; i < ja.size(); i++)
					messages[i] = ja.get(i).getAsString();
			} else messages = new String[0];

            // Create ApiResponse with additional fields if needed
            ApiResponse response = new ApiResponse(status, token, messages, chatEnded, chatState, locale, chatbotWaiting);

            return response;
        } catch (Exception e) {
            System.err.println("Failed to parse JSON response: " + e.getMessage());
            return null;
        }
    }

    /**
     * Utility method to check if currently logged in
     * @return True if logged in
     */
    public boolean isLoggedIn() {
        return serverToken != null && !serverToken.isEmpty();
    }

    /**
     * Gets the current token
     * @return Current token
     */
    public String getCurrentToken() {
        return serverToken;
    }

    /**
     * Manual token management for advanced scenarios
     * @param token New token value
     */
    public void setToken(String token) {
    	serverToken = token;
    }

    /**
     * Direct JSON parsing utility method
     * @param json JSON string
     * @return ApiResponse if successful, null otherwise
     */
    public ApiResponse parseJsonResponse(String json) {
        return parseResponseWithAdditionalFields(json);
    }

    ///////////////////////////////////////////////////////////////////////////

    /**
     * ApiResponse class using Gson annotations
     */
    public static class ApiResponse {
        @SerializedName("STATUS")
        private String status; // OK or ERROR

        @SerializedName("TOKEN")
		private String token; // Exists only on login response

        @SerializedName("MESSAGES")
        private String[] messages; // Array of messages from chatbot, usually 1 message

        @SerializedName("CHAT_ENDED")
        private boolean chatEnded; // True if the chat has ended and the client is disconnected

        @SerializedName("CHAT_STATE")
        private String chatState; // Current chat state. Could be either CHATBOT, AIMODEL or AGENT

        @SerializedName("LOCALE")
        private String locale; // Current chat session locale

        @SerializedName("CHATBOT_WAITING")
        private boolean chatbotWaiting; // True if the chatbot is waiting for user input, use when switching to AIMODEL (Send '...' back).

        // Default constructor for Gson
        public ApiResponse() {}

        /**
         * ApiResponse constructor
         * @param status Response status
         * @param token Response token
         * @param messages Array of messages from chatbot
         * @param chatEnded True if the chat has ended and the client is disconnected
         * @param chatState Current chat state. Could be either CHATBOT, AIMODEL or AGENT
         * @param locale Current chat session locale
         */
		public ApiResponse(String status, String token, String[] messages, boolean chatEnded, String chatState, String locale, boolean chatbotWaiting) {
            this.status = status;
			this.token = token;
			this.messages = messages;
			this.chatEnded = chatEnded;
			this.chatState = chatState;
			this.locale = locale;
			this.chatbotWaiting = chatbotWaiting;
        }

		public String getToken() { return token; } // Return communication token
        public boolean isSuccess() { return "OK".equals(status); } // Return true if status is OK
		public boolean isChatEnded() { return chatEnded; } // Return true if chat has ended
		public boolean isChatbotWaiting() { return chatbotWaiting; } // Return true if chatbot is waiting for user input
		public String getChatState() { return chatState; } // Return current chat state
		public String[] getMessages() { return messages; } // Return array of messages from chatbot
		public String getLocale() { return locale; } // Return current chat session locale

		/**
		 * Get chat state icon name based on current chat state
		 * @return Chat state icon name
		 */
		public String getChatStateIconName() {
			if (chatState != null) {
				if(chatState.equals("AIMODEL")) return "assistant";
				if(chatState.equals("AGENT")) return "agent";
				if(chatState.equals("CHATBOT")) return "chatbot";
			}
			return null;
		}

		@Override
		public String toString() {
			return "ApiResponse{" +
					"status='" + status + '\'' +
					", token='" + token + '\'' +
					", messages=" + Arrays.toString(messages) +
					", chatEnded=" + chatEnded +
					", chatState='" + chatState + '\'' +
					", locale='" + locale + '\'' +
					", chatbotWaiting=" + chatbotWaiting +
					'}';
		}
    }
}